import matplotlib.pyplot as plt
import os


class Param:
    def __init__(self, names, values):
        if len(names) != len(values):
            raise ValueError("Length of names and values must be equal")

        self.names = names
        self.values = values

    def get_param(self, param_name):
        index = self.names.index(param_name)
        return self.values[index]

    def __str__(self):
        result = ""
        for name, value in zip(self.names, self.values):
            result += f" {name} = {value},"

        return result[:-1] if len(result) > 0 else ""


def clean_path(path: str):
    if path.endswith("/"):
        return path[:-1]

    return path


def plot_optimizers_result(
        optimizers,
        params,
        value_distances_per_param,
        d_estimation_error_per_param,
        model_name,
        save=False,
        plots_directory=None,
        plot_every=1,
        mark_every=50):

    print(f"Mark every {mark_every} steps")
    if save and plots_directory is None:
        raise ValueError("Should specify path to save the plots")

    markers = ['s', 'o', 'v', 'P', 'X', 'D']
    colors = ['blue', 'orange', 'green', 'red', 'purple', 'lightseagreen']

    os.makedirs(plots_directory, exist_ok=True)

    num_params = len(params)
    fig, axes = plt.subplots(1, num_params, figsize=(5 * num_params, 4))  # 1 row, num_params columns for subplots

    # First plot: value distances
    for j, param in enumerate(params):
        ax = axes[j] if len(params) > 1 else axes
        for i, (optimizer, model) in enumerate(optimizers):
            optimizer_name = optimizer.__class__.__name__
            values = [v if v > 1e-14 else 1e-14 for v in value_distances_per_param[param][optimizer_name][::plot_every]]
            ax.plot(
                values,
                label=optimizer_name,
                marker=markers[i],
                color=colors[i],
                linewidth=1,
                markersize=4,
                markevery=mark_every)

        ax.set_title(f"${str(param)}$")
        ax.set_xlabel("Number of oracle calls")
        ax.set_ylabel(r'$f(x_k^*) - f^*$')
        ax.set_yscale('log')
        ax.legend()
        ax.grid(True)

    # Adjust layout for better spacing
    fig.tight_layout()

    # Save or show the plot
    if save:
        plt.savefig(f"{clean_path(plots_directory)}/comparison-{model_name}-residual.pdf", transparent=True)

    plt.show(block=False)

    fig, axes = plt.subplots(1, num_params, figsize=(5 * num_params, 4))  # 1 row, num_params columns for subplots

    # Second plot: estimation error
    for j, param in enumerate(params):
        ax = axes[j] if len(params) > 1 else axes
        for i, (optimizer, model) in enumerate(optimizers):
            optimizer_name = optimizer.__class__.__name__
            if not optimizer.has_d_estimator():
                continue
            ax.plot(
                d_estimation_error_per_param[param][optimizer_name][::plot_every],
                label=optimizer_name,
                marker=markers[i],
                color=colors[i],
                linewidth=1,
                markersize=4,
                markevery=mark_every)

        ax.set_title(f"${str(param)}$")
        ax.set_xlabel("Number of oracle calls")
        ax.set_ylabel(r'$\frac{D_{0}}{\bar{r}}$')
        ax.set_yscale('log')
        ax.legend()
        ax.grid(True)

    # Adjust layout for better spacing
    fig.tight_layout()

    # Save or show the plot
    if save:
        plt.savefig(f"{clean_path(plots_directory)}/comparison-{model_name}-estimate-error.pdf", transparent=True)

    plt.show(block=False)


def run_different_opts(model_optimizer_pair, iterations, optimal_point, log_per, noise_levels=None):
    if noise_levels is None:
        noise_levels = [0]
    value_distances = {}
    point_distances = {}

    for j, nl in enumerate(noise_levels):
        print(f"Noise STD = {nl}")

        for i, (optimizer, model) in enumerate(model_optimizer_pair):
            optimizer_name = optimizer.__class__.__name__
            print(f"{optimizer_name} Method")
            model.set_optimizer(optimizer)
            point_distance_list, value_distance_list = model.optimize(optimal_point=optimal_point,
                                                                      max_iter=iterations,
                                                                      log_per=log_per)
            print()
            print()

            value_distances[optimizer_name] = value_distance_list
            point_distances[optimizer_name] = point_distance_list

    return point_distances, value_distances
